import React, { Component } from 'react';
import { withStyles } from '@material-ui/core/styles';
import { modelToName } from 'library/utils/ModelUtils';
import InfoTooltip from 'components/InfoTooltip';
import styles from './styles';

type Props = {
    children: ReactElement,
    showInfo: object,
    language: string,
    strings: object,
    hasBetaVersion: boolean,
    betaVersion: string,
    classes: {
        [key: string]: string
    }
};

class HeaderLabel extends Component<Props> {
    constructor(props) {
        super(props);
        this.state = {
            device: props.showInfo,
            language: props.language,
            strings: props.strings
        };
    }

    static getDerivedStateFromProps(nextProps, prevState) {
        const { language, device } = prevState;
        if (language !== nextProps.language) {
            return {
                language: nextProps.language,
                strings: nextProps.strings
            };
        }
        if (
            device &&
            nextProps.showInfo &&
            (prevState.device.connected !== nextProps.showInfo.connected ||
                prevState.device.fwVersion !== nextProps.showInfo.fwVersion)
        ) {
            return { device: nextProps.showInfo };
        }
        return null;
    }

    renderTooltip() {
        const { strings, device } = this.state;
        const { classes, hasBetaVersion, betaVersion } = this.props;
        const isBetaInstalled = !!(
            betaVersion && betaVersion.toString() === device.fwVersion
        );
        return (
            <div className={classes.tooltipInnerContainer}>
                <div className={classes.tooltipColumn}>
                    <div className={classes.tooltipLabel}>
                        {strings.device_info_model}
                    </div>
                    <div className={classes.tooltipLabel}>
                        {strings.device_info_current_fw}
                    </div>
                    <div className={classes.tooltipLabel}>
                        {strings.device_info_serial}
                    </div>
                </div>
                <div className={classes.tooltipColumn}>
                    <div className={classes.tooltipValue}>
                        {device.model ? modelToName(device.model) : ''}
                    </div>
                    <div className={classes.tooltipValue}>
                        {device.fwVersion}{' '}
                        {hasBetaVersion && isBetaInstalled && device.btmVersion
                            ? `(${device.btmVersion})`
                            : ''}
                    </div>
                    <div className={classes.tooltipValue}>{device.serial}</div>
                </div>
            </div>
        );
    }

    render() {
        const { device } = this.state;
        const { children, classes } = this.props;
        return (
            <div className={classes.container}>
                <div className={classes.title}>{children}</div>
                {device && (
                    <InfoTooltip placement="bottom-start">
                        {this.renderTooltip()}
                    </InfoTooltip>
                )}
            </div>
        );
    }
}

export default withStyles(styles)(HeaderLabel);
